'''October 29th 2018
A set of functions to convert the spike times and finely sampled angles/
positions from the data into rates and smoothed angles/positions, to be 
used in further decoding. These rates are saved and then accessed through
the binned_spikes_class functions.
Two methods to get rates: either using non-overlapping bins or convolving with 
a Gaussian kernel. The non-overlapping bins method is currently not being
used (and won't work), but the option is here for historical reasons.
'''

from __future__ import division
import numpy as np
import sys, os

gen_fn_dir = os.path.abspath('../shared_scripts/')
sys.path.append(gen_fn_dir)

from angle_fns import circmean

def get_rates_and_angles_by_interval(inp_data, params, smooth_type='kernel', just_wake=True):
    '''Extracts rates/spike counts and smoothed angles from the data file
    passed in inp_data (in turn generated by get_processed_data). Return a dict
    with keys: cells, session and states. 
    States is a nested dict. State->intervals->(1)num_spikes, (2)bin_edges, (3)avg_angles
    num_spikes consists of a dict of cell IDs and rates/spike counts in appropriate bins.'''
    
    session = inp_data['session']
    spike_times = inp_data['spike_times']
    cell_IDs = sorted(inp_data['spike_times'].keys())
    if just_wake:
        states = ['Wake']
    else:
        states = inp_data['state_times'].keys()

    out_data = {s: {} for s in states}
    out_data.update({'session': session, 'cells': cell_IDs})

    if smooth_type == 'kernel':
        out_data['params'] = params
    elif smooth_type == 'binned':
        print 'This option is here for historical reasons. May add later'
        return False
        win_size = params
        out_data['win_size'] = win_size
    else:
        print 'Unknown smoothing type'
        return False

    for state in states:
        for i, tmp_interval in enumerate(inp_data['state_times'][state]):
            interval = tuple(tmp_interval)
            print state, interval
            out_data[state][interval] = {}
            if smooth_type == 'kernel':
                rate_times, rates, angle_times, angles = get_rates_angles_kernel_dict(
                    inp_data, params, interval)
                out_data[state][interval]['rates'] = rates
                out_data[state][interval]['rate_times'] = rate_times
                out_data[state][interval]['angles'] = angles
                out_data[state][interval]['angle_times'] = angle_times
            elif smooth_type == 'binned':
                spike_counts, bin_edges, avg_angles = get_rates_angles_binned_dict(inp_data, 
                    interval[0], interval[1], win_size)
                out_data[state][interval]['num_spikes'] = spike_counts
                out_data[state][interval]['bin_edges'] = bin_edges
                out_data[state][interval]['avg_angles'] = avg_angles
    return out_data

def get_rates_angles_kernel_dict(inp_data, params, interval):
    '''Convert the set of spike_times into rates using time_bins. Assume that spike_times
    is a dictionary with keys being the cells and values the spike times. 
    Params contains sigma, dt.'''

    if params['method'] == 'gaussian':
        wind_fn = lambda mu, x: gaussian_wind_fn(mu, params['sigma'], x)
    else:
        print 'Unknown windowing function'

    spike_times = inp_data['spike_times']
    angle_list = inp_data['angle_list']
    samp_rate = inp_data['pos_sampling_rate']
    dt = params['dt']

    bin_edges = np.arange(interval[0], interval[1], dt)
    time_vals = bin_edges[:-1] + (dt / 2.)
    cell_IDs = spike_times.keys()

    # For each cell, pull out the appropriate spike times and compute rates
    interval_spike_times = {}
    rates = {}

    for y in cell_IDs:
        print y
        interval_spike_times[y] = [x for x in spike_times[y]
                                   if interval[0] <= x < interval[1]]
        rates[y] = get_kernel_sum(interval_spike_times[y], time_vals, wind_fn)

    angle_idx = np.round(bin_edges * samp_rate).astype(int)    
    # interval_times = zip(bin_edges[:-1], bin_edges[1:])
    # interval_idx = [(np.rint(x[0] * samp_rate).astype(int),
    #                  np.rint(x[1] * samp_rate).astype(int)) for x in interval_times]

    sm_angles = [circmean(angle_list[i0:i1]) for i0,i1 in zip(angle_idx[:-1], 
        angle_idx[1:])]

    # Note change in order that we're returning values
    # used to be "return sm_angles, rates, time_vals, time_vals"
    return time_vals, rates, time_vals, np.array(sm_angles)

def gaussian_wind_fn(mu, sigma, x):
    '''Normalized Gaussian'''
    return np.exp(-((x - mu)**2) / (2 * sigma**2)) / (sigma * np.sqrt(2 * np.pi))

def get_kernel_sum(spike_list, t_points, win_fun):
    '''Sum a bunch of kernels centered at each spike time. This can be slow, but 
    is only run once at the beginning so not optimizing.
    '''
    result = np.zeros_like(t_points)
    for spike in spike_list:
        result = result + win_fun(spike, t_points)
    return result
